/**
  ******************************************************************************
  * @file    usb_mem.c
  * @author  none
  * @version V4.1.0
  * @date    17-Oct-2024
  * @brief   Utility functions for memory transfers to/from PMA
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; copyright (c) 2024 WIZnet. 
  * All rights reserved.</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without 
  * modification, are permitted, provided that the following conditions are met:
  *
  * 1. Redistribution of source code must retain the above copyright notice, 
  *    this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright notice,
  *    this list of conditions and the following disclaimer in the documentation
  *    and/or other materials provided with the distribution.
  * 3. Neither the name of WIZnet nor the names of other 
  *    contributors to this software may be used to endorse or promote products 
  *    derived from this software without specific written permission.
  * 4. This software, including modifications and/or derivative works of this 
  *    software, must execute solely and exclusively on microcontroller or
  *    microprocessor devices manufactured by or for WIZnet.
  * 5. Redistribution and use of this software other than as permitted under 
  *    this license is void and will automatically terminate your rights under 
  *    this license. 
  *
  * THIS SOFTWARE IS PROVIDED BY WIZnet AND CONTRIBUTORS "AS IS" 
  * AND ANY EXPRESS, IMPLIED OR STATUTORY WARRANTIES, INCLUDING, BUT NOT 
  * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
  * PARTICULAR PURPOSE AND NON-INFRINGEMENT OF THIRD PARTY INTELLECTUAL PROPERTY
  * RIGHTS ARE DISCLAIMED TO THE FULLEST EXTENT PERMITTED BY LAW. IN NO EVENT 
  * SHALL WIZnet OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
  * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
  * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
  * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
  * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "usb_lib.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Extern variables ----------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/**
  * Function Name  : UserToPMABufferCopy
  * Description    : Copy a buffer from user memory area to packet memory area (PMA)
  * Input          : - pbUsrBuf: pointer to user memory area.
  *                  - wPMABufAddr: address into PMA.
  *                  - wNBytes: no. of bytes to be copied.
  * Output         : None.
  * Return         : None	.
  **/
void UserToPMABufferCopy(uint8_t *pbUsrBuf, uint16_t wPMABufAddr, uint16_t wNBytes)
{
#if defined mhF303xE || defined mhF302x8
    uint32_t  n = (wNBytes + 1) >> 1; /* n = (wNBytes + 1) / 2 */
    uint32_t  i;
    uint16_t *pdwVal;
    pdwVal = (uint16_t *)(wPMABufAddr + PMAAddr);

    for (i = n; i != 0; i--)
    {
        *pdwVal++ = *(uint16_t *)pbUsrBuf++;
        pbUsrBuf++;
    }
#else
    uint32_t  n = (wNBytes + 1) >> 1; /* n = (wNBytes + 1) / 2 */
    uint32_t  i, temp1, temp2;
    uint16_t *pdwVal;
    pdwVal = (uint16_t *)(wPMABufAddr * 2 + PMAAddr);
    for (i = n; i != 0; i--)
    {
        temp1 = (uint16_t)*pbUsrBuf;
        pbUsrBuf++;
        temp2     = temp1 | (uint16_t)*pbUsrBuf << 8;
        *pdwVal++ = temp2;
        pdwVal++;
        pbUsrBuf++;
    }
#endif
}

/**
  * Function Name  : PMAToUserBufferCopy
  * Description    : Copy a buffer from user memory area to packet memory area (PMA)
  * Input          : - pbUsrBuf    = pointer to user memory area.
  *                  - wPMABufAddr = address into PMA.
  *                  - wNBytes     = no. of bytes to be copied.
  * Output         : None.
  * Return         : None.
  **/
void PMAToUserBufferCopy(uint8_t *pbUsrBuf, uint16_t wPMABufAddr, uint16_t wNBytes)
{
#if defined mhF303xE || defined mhF302x8
    uint32_t  n = (wNBytes + 1) >> 1; /* /2*/
    uint32_t  i;
    uint16_t *pdwVal;
    pdwVal = (uint16_t *)(wPMABufAddr + PMAAddr);
    for (i = n; i != 0; i--)
    {
        *(uint16_t *)pbUsrBuf++ = *pdwVal++;
        pbUsrBuf++;
    }
#else
    uint32_t  n = (wNBytes + 1) >> 1; /* /2*/
    uint32_t  i;
    uint32_t *pdwVal;
    pdwVal = (uint32_t *)(wPMABufAddr * 2 + PMAAddr);
    for (i = n; i != 0; i--)
    {
        *(uint16_t *)pbUsrBuf++ = *pdwVal++;
        pbUsrBuf++;
    }
#endif
}

/************************ (C) COPYRIGHT  2024  WIZnet *****END OF FILE****/
