#include "bsp_i2c_ee.h"
#include "bsp_i2c_gpio.h"

/*
*********************************************************************************************************
* Function name: i2c_CheckDevice
* Function description: Detect the I2C bus device, the CPU sends the address of the sending device, and then reads the device reply to determine whether the device exists
* Parameters: _Address: I2C bus address of the device
* Return value: Return value 0 means correct, return 1 means not detected
*********************************************************************************************************
*/
uint8_t ee_CheckDevice(uint8_t _Address)
{
    uint8_t ucAck;

    i2c_Start(); /* Send start signal */

    /* Send device address + read and write control bits (0 = w, 1 = r) bit7 first */
    i2c_SendByte(_Address | EEPROM_I2C_WR);
    ucAck = i2c_WaitAck(); /* Detect the ACK response of the device */

    i2c_Stop();            /* Send stop signal */

    i2c_NAck();            /*If the input is a read address, a negative-acknowledgement signal needs to be generated*/

    return ucAck;
}


/*
*********************************************************************************************************
* Function name: ee_WaitStandby
* Function description: Wait for the EEPROM to be ready. After writing data, this function must be called

During the write operation, after transferring the data to the EEPROM using I2C,
EEPROM will take a certain amount of time to write data to the internal space,
When the internal write of the EEPROM is completed, it will respond to the addressing of the I2C device.
Call this function to wait until the EEPROM internal timing is written
* Parameters: None
* Return value: 0 means normal, 1 means wait timeout
*********************************************************************************************************
*/
uint8_t ee_WaitStandby(void)
{
    uint32_t wait_count = 0;

    while (ee_CheckDevice(EEPROM_DEV_ADDR))
    {
        //If the number of detections exceeds, exit the loop
        if (wait_count++ > 0xFFFF)
        {
            //wait timeout
            return 1;
        }
    }
    //Wait for completion
    return 0;
}

/*
*********************************************************************************************************
* Function name: ee_ReadBytes
* Function description: Start reading several data from the address specified by the serial EEPROM
* Parameter: _usAddress: Start address
* _usSize: Data length in bytes
* _pReadBuf: Buffer pointer to hold read data
Return value: 0 indicates failure, 1 indicates success
*********************************************************************************************************
*/
uint8_t ee_ReadBytes(uint8_t *_pReadBuf, uint16_t _usAddress, uint16_t _usSize)
{
    uint16_t i;

    /* Use serial EEPROM to read the instruction sequence randomly, continuously reading several bytes */

    /* Step 1: Initiate the I2C bus start signal */
    i2c_Start();

    /* Step 2: Initiate the control byte, the high 7bit is the address, bit0 is the read and write control bit, 0 means write, 1 means read */
    i2c_SendByte(EEPROM_DEV_ADDR | EEPROM_I2C_WR); /* Here is the write instruction */

    /* Step 3: Wait for ACK */
    if (i2c_WaitAck() != 0)
    {
        goto cmd_fail; /* EEPROM device does not respond */
    }

    /* Step 4: Send the byte address, 24C02 is only 256 bytes, so 1 byte is enough. If it is above 24C04, then multiple addresses need to be sent in a row here */
    i2c_SendByte((uint8_t)_usAddress);

    /* Step 5: Wait for ACK */
    if (i2c_WaitAck() != 0)
    {
        goto cmd_fail; /* EEPROM device does not respond */
    }

    /* Step 6: Restart the I2C bus. The purpose of the previous code is to transfer the address to the EEPROM, and the following starts reading the data */
    i2c_Start();

    /* Step 7: Initiate the control byte, the high 7bit is the address, bit0 is the read and write control bit, 0 means write, 1 means read */
    i2c_SendByte(EEPROM_DEV_ADDR | EEPROM_I2C_RD); /* Here is the read command */

    /* Step 8: Send ACK */
    if (i2c_WaitAck() != 0)
    {
        goto cmd_fail; /* EEPROM device does not respond */
    }

    /* Step 9: Loop through the data */
    for (i = 0; i < _usSize; i++)
    {
        _pReadBuf[i] = i2c_ReadByte(); /* Read 1 byte */

        /* After reading 1 byte, you need to send Ack, the last byte does not need Ack, send Nack */
        if (i != _usSize - 1)
        {
            i2c_Ack(); /* After the intermediate byte is read, the CPU generates an ACK signal (driver SDA = 0) */
        }
        else
        {
            i2c_NAck(); /* After the last byte is read, the CPU generates a NACK signal (driver SDA = 1) */
        }
    }
    /* Send I2C bus stop signal */
    i2c_Stop();
    return 1; /* Execution successful */

cmd_fail:     /* After the command execution fails, remember to send a stop signal to avoid affecting other devices on the I2C bus */
    /* Send I2C bus stop signal */
    i2c_Stop();
    return 0;
}

/*
*********************************************************************************************************
* Function name: ee_WriteBytes
* Function description: Write some data to the specified address of the serial EEPROM, and use the page write operation to improve the writing efficiency
* Parameter: _usAddress: Start address
* _usSize: Data length in bytes
* _pWriteBuf: Buffer pointer to hold read data
Return value: 0 indicates failure, 1 indicates success
*********************************************************************************************************
*/
uint8_t ee_WriteBytes(uint8_t *_pWriteBuf, uint16_t _usAddress, uint16_t _usSize)
{
    uint16_t i, m;
    uint16_t usAddr;

    /*
	Unlike read operations, which can read many bytes in a row, write serial EEPROMs can only be written on the same page each time.
	For 24xx02, page size = 8
	The simple processing method is: according to the byte write operation mode, every byte written, the address is sent
	In order to improve the efficiency of continuous writing: This function uses the page wirte operation.
	*/

    usAddr = _usAddress;
    for (i = 0; i < _usSize; i++)
    {
        /* When sending the first byte or the first page address, you need to restart the start signal and address */
        if ((i == 0) || (usAddr & (EEPROM_PAGE_SIZE - 1)) == 0)
        {
            /* Step 0: Send a stop signal to start the internal write operation */
            i2c_Stop();

            /* Determine whether the internal write operation is completed by checking the response of the device, generally less than 10ms 	
			
			  When the CLK frequency is 200KHz, the number of queries is about 30 times
			The principle is the same as ee_WaitStandby function, but the function will generate a stop signal after the check is completed, which does not apply here
			*/
            for (m = 0; m < 1000; m++)
            {
                /* Step 1: Initiate the I2C bus start signal */
                i2c_Start();

                /* Step 2: Initiate the control byte, the high 7bit is the address, bit0 is the read and write control bit, 0 means write, 1 means read */
                i2c_SendByte(EEPROM_DEV_ADDR | EEPROM_I2C_WR); /* Here are the write instructions */

                /* Step 3: Send a clock to determine if the device is responding correctly */
                if (i2c_WaitAck() == 0)
                {
                    break;
                }
            }
            if (m == 1000)
            {
                goto cmd_fail; /* EEPROM device write timeout */
            }

            /* Step 4: Send the byte address, 24C02 is only 256 bytes, so 1 byte is enough. If it is above 24C04, then multiple addresses need to be sent in a row here */
            i2c_SendByte((uint8_t)usAddr);

            /* Step 5: Wait for ACK */
            if (i2c_WaitAck() != 0)
            {
                goto cmd_fail; /* EEPROM device does not respond */
            }
        }

        /* Step 6: Start writing data */
        i2c_SendByte(_pWriteBuf[i]);

        /* Step 7: Send ACK */
        if (i2c_WaitAck() != 0)
        {
            goto cmd_fail; /* EEPROM device does not respond */
        }

        usAddr++; /* Add 1 to the address */
    }

    /* The command is executed successfully, and the I2C bus stop signal is sent */
    i2c_Stop();

    //Wait for the last internal EEPROM write to complete
    if (ee_WaitStandby() == 1) //Equal to 1 means timeout
        goto cmd_fail;

    return 1;

cmd_fail: /* After the command execution fails, remember to send a stop signal to avoid affecting other devices on the I2C bus */
    /* Send I2C bus stop signal */
    i2c_Stop();
    return 0;
}


void ee_Erase(void)
{
    uint16_t i;
    uint8_t  buf[EEPROM_SIZE];

    /* fill buffer */
    for (i = 0; i < EEPROM_SIZE; i++)
    {
        buf[i] = 0xFF;
    }

    /* Write EEPROM, start address = 0, data length is 256 */
    if (ee_WriteBytes(buf, 0, EEPROM_SIZE) == 0)
    {
        printf("Error erasing eeprom!\r\n");
        return;
    }
    else
    {
        printf("Erased eeprom successfully!\r\n");
    }
}


/*
* eeprom AT24C02 read and write test
* Normal returns 1, abnormal returns 0
*/
uint8_t ee_Test(void)
{
    uint16_t i;
    uint8_t  write_buf[EEPROM_SIZE];
    uint8_t  read_buf[EEPROM_SIZE];

    /*-----------------------------------------------------------------------------------*/
    if (ee_CheckDevice(EEPROM_DEV_ADDR) == 1)
    {
        /* No EEPROM detected */
        printf("No serial EEPROM detected!\r\n");

        return 0;
    }
    /*------------------------------------------------------------------------------------*/
    /* Fill the test buffer */
    for (i = 0; i < EEPROM_SIZE; i++)
    {
        write_buf[i] = i;
    }
    /*------------------------------------------------------------------------------------*/
    if (ee_WriteBytes(write_buf, 0, EEPROM_SIZE) == 0)
    {
        printf("Error writing eeprom!\r\n");
        return 0;
    }
    else
    {
        printf("Write eeprom successfully!\r\n");
    }

    /*-----------------------------------------------------------------------------------*/
    if (ee_ReadBytes(read_buf, 0, EEPROM_SIZE) == 0)
    {
        printf("Error reading eeprom!\r\n");
        return 0;
    }
    else
    {
        printf("Read eeprom successfully, the data is as follows:\r\n");
    }
    /*-----------------------------------------------------------------------------------*/
    for (i = 0; i < EEPROM_SIZE; i++)
    {
        if (read_buf[i] != write_buf[i])
        {
            printf("0x%02X ", read_buf[i]);
            printf("Error: EEPROM readout does not match the data written");
            return 0;
        }
        printf(" %02X", read_buf[i]);

        if ((i & 15) == 15)
        {
            printf("\r\n");
        }
    }
    printf("Eeeprom read and write test was successful\r\n");
    return 1;
}
/*********************************************END OF FILE**********************/
